//Conventions:
//    Global constants (declared with const) and #defines - all uppercase letters with words separated 
//        by underscores.
//        (E.G., #define MY_DEFINE 5).
//        (E.G., const int MY_CONSTANT = 5;).
//    New data types (classes, structs, typedefs, etc.) - begin with an uppercase letter followed by
//        lowercase words separated by uppercase letters.  Enumerated constants contain a prefix
//        associating them with a particular enumerated set.
//        (E.G., typedef int MyTypedef;).
//        (E.G., enum MyEnumConst {MEC_ONE, MEC_TWO};)
//    Global variables - begin with "g_" followed by lowercase words separated by underscores.
//        (E.G., int g_my_global;).
//    Local variables - begin with a lowercase letter followed by lowercase words separated by
//        underscores.
//        (E.G., int my_local;).
//    Argument variables - begin with "a_" followed by lowercase words separated by underscores.
//        (E.G., ...int a_my_argument, ...).
//    Member variables - begin with "m_" followed by lowercase words separated by underscores.
//        (E.G., int m_my_member;).
//    Functions (member or global) - begin with an uppercase letter followed by lowercase words
//        separated by uppercase letters.
//        (E.G., void MyFunction(void);).


//******************************************************************************************************
//**** SYSTEM HEADER FILES
//******************************************************************************************************
#include <PlaySound.h>


//******************************************************************************************************
//**** PROJECT HEADER FILES
//******************************************************************************************************
#include "EventAddOnPublic.h"	//Scheduler add-on event handler API
#include "MessageWithSound.h"
#include "Colors.h"


//******************************************************************************************************
//**** GLOBAL VARIABLES AND CONSTANTS
//******************************************************************************************************
//The sound file
sem_id g_sound_sem;
char* g_sound_file;


//******************************************************************************************************
//**** FUNCTION DEFINITIONS
//******************************************************************************************************
char* InitializeAddOn()
{
	//Create the semaphore
	if ((g_sound_sem = create_sem(1,"My Semaphore")) < B_NO_ERROR)
		return NULL;

	//Read the user sound selection from the preferences
	char* data;
	ssize_t data_size;
	uint32 data_type;
	g_sound_file = NULL;
	if(ReadAddOnSetting("MessageWithSound_Sound",data,data_size,data_type) == B_OK)
 		if(data_type == B_INT8_TYPE && data_size >= 1)
 		{
			g_sound_file = new char[strlen(data)+1];
			strcpy(g_sound_file,data);
		}
	return "Message with sound";
}


void CleanUpAddOn()
{
	if(g_sound_file)
	{
		WriteAddOnSetting("MessageWithSound_Sound",g_sound_file,strlen(g_sound_file)+1,B_INT8_TYPE);
		delete[] g_sound_file;
	}
	delete_sem(g_sound_sem);
}


EventAddOnData* NewAddOnData(Event* a_attach_to,BufferedFile* a_file,bool* a_file_ok)
{
	return new MessageSoundDataHandler(a_attach_to, a_file, a_file_ok);
}


//******************************************************************************************************
//**** THE EVENT HANDLER
//******************************************************************************************************
MessageSoundDataHandler::MessageSoundDataHandler(Event *a_attach_to, BufferedFile *a_file, bool *a_file_ok)
: EventAddOnData(a_attach_to)
{
	m_message = NULL;

	//Read the data
	*a_file_ok = true;
	if(a_file)
	{
		int32 data_length;
		if(a_file->ReadNumber(&data_length,sizeof(uint32),1) != 1)
			{a_file_ok = false; return;}
		if(data_length != 0)
		{
			m_message = new char[data_length];
			if(a_file->ReadRaw(m_message,data_length) != data_length)
				{a_file_ok = false; return;}
		}
	}
}


MessageSoundDataHandler::~MessageSoundDataHandler()
{
	if(m_message)
		delete[] m_message;
}


BView* MessageSoundDataHandler::CreateEditView(BRect a_frame)
{
	return new MessageSoundEditView(a_frame,this);
}


bool MessageSoundDataHandler::SaveData(BufferedFile *a_file)
{
	int32 data_length = 0;
	if(m_message)
		data_length = strlen(m_message) + 1;
	if(a_file->WriteNumber(&data_length,sizeof(data_length)) != 1)
		return false;
	if(data_length != 0)
		if(a_file->WriteRaw(m_message,data_length) != data_length)
			return false;
	return true;
}


void MessageSoundDataHandler::DealWithEvent(bool a_treat_as_late)
{
	//Find out if I need to ask whether to nag or not
	AssertableRWSemaphore* protection = GetAttachedTo()->GetProtectionSemaphore();
	protection->ReadLock();
	bool ask_nag = GetAttachedTo()->GetApplyNag();
	protection->ReadUnlock();

	//For demonstration purposes, not really functional: if the thread I am about to spawn were to use any information
	//from the MessageSoundDataHandler, I would need to pass it its own copy that it will delete, since it can't rely on the
	//Event (or MessageSoundDataHandler instance) remaining intact, because the user may have the delete once delivered flag
	//set.
	char* temp_data = NULL;
	if(m_message)
	{
		temp_data = new char[strlen(m_message)+1];
		strcpy(temp_data,m_message);
	}

	//Start the sound
	thread_id sound_thread = spawn_thread(PlayMessageSound,"MessageSound",B_NORMAL_PRIORITY,temp_data);
	if(sound_thread != B_NO_MORE_THREADS && sound_thread != B_NO_MEMORY)
	{
		EventAddOnData::ProcessHold();
		resume_thread(sound_thread);
	}
	else if(temp_data)
		delete[] temp_data;

	//Ask the question
	if(ask_nag)
	{
		if(DeliverEventMessage(GetAttachedTo(),a_treat_as_late,"\n\n",m_message,NULL,NULL,NULL,"Don't nag","OK") == 0)
			GetAttachedTo()->SetSkipNag();
	}
	else
		DeliverEventMessage(GetAttachedTo(),a_treat_as_late,"\n\n",m_message,NULL,NULL,NULL,"OK",NULL);
}


int32 PlayMessageSound(void *data)
{
	char* thread_data = (char*)data;
	//Don't actually use this data, it was just for demonstration; see above
	if(thread_data)
		delete[] thread_data;

	if (acquire_sem(g_sound_sem) == B_NO_ERROR)
	{
		if(g_sound_file)
		{
			BEntry sound_entry(g_sound_file);
			release_sem(g_sound_sem);
			entry_ref sound_ref;
			if(sound_entry.InitCheck() == B_NO_ERROR && sound_entry.Exists() && sound_entry.GetRef(&sound_ref) == B_NO_ERROR)
				play_sound(&sound_ref, true, true, false);
		}
		else
			release_sem(g_sound_sem);
	}
	EventAddOnData::ProcessDone();
	return 0;
}


void MessageSoundDataHandler::SetMessage(const char *a_new_message)
{
	if(m_message)
		delete[] m_message;
	if(a_new_message && a_new_message[0] != 0)
	{
		m_message = new char[strlen(a_new_message)+1];
		strcpy(m_message,a_new_message);
	}
	else
		m_message = NULL;
	AssertableRWSemaphore* protection = GetAttachedTo()->GetProtectionSemaphore();
	protection->ReadWriteLock();
	GetAttachedTo()->SetModified();
	protection->ReadWriteUnlock();
}


const char* MessageSoundDataHandler::GetMessage()
{
	return m_message;
}


MessageSoundEditView::MessageSoundEditView(BRect a_frame, MessageSoundDataHandler *a_attach_to)
: BView(a_frame,NULL,B_FOLLOW_ALL_SIDES,B_WILL_DRAW)
{
	SetViewColor(BeBackgroundGrey);

	//Add selection button
	BRect bounds = Bounds();
	m_button = new BButton(BRect(-1,-1,-1,-1),NULL,"Select sound",new BMessage(MSG_SET_SOUND),B_FOLLOW_LEFT|B_FOLLOW_BOTTOM);
	float width,height;
	m_button->GetPreferredSize(&width,&height);
	m_button->ResizeTo(width,height);
	m_button->MoveTo(2.0,bounds.bottom-height);
	AddChild(m_button);

	//Create the text view
	bounds.bottom -= height+4;
	bounds.left += 1;
	bounds.top += 1;
	bounds.right -= B_V_SCROLL_BAR_WIDTH+1;
	bounds.bottom -= 1;
	MessageSoundEditTextView* text_view = new MessageSoundEditTextView(bounds,a_attach_to);
	AddChild(new BScrollView(NULL,text_view,B_FOLLOW_ALL_SIDES,0,false,true,B_PLAIN_BORDER));
	m_file_panel = NULL;
}


MessageSoundEditView::~MessageSoundEditView()
{
	delete m_file_panel;
}


void MessageSoundEditView::AttachedToWindow()
{
	m_button->SetTarget(this);

	//Create file panel
	BMessenger messenger(this);
	entry_ref *sound_ref = NULL;
	if (acquire_sem(g_sound_sem) == B_NO_ERROR)
	{
		if(g_sound_file)
		{
			BEntry sound_entry(g_sound_file);
			sound_entry.GetParent(&sound_entry);
			entry_ref temp_entry;
			if(sound_entry.GetRef(&temp_entry) == B_NO_ERROR)
				sound_entry = &temp_entry;
		}
		release_sem(g_sound_sem);
	}
	m_file_panel = new BFilePanel(B_OPEN_PANEL,&messenger,sound_ref,0,false); 
}


void MessageSoundEditView::MessageReceived(BMessage* a_message)
{
	if(a_message->what == MSG_SET_SOUND)
		m_file_panel->Show();
	else if(a_message->what == B_REFS_RECEIVED)
	{
		entry_ref new_sound_ref;
		if(a_message->FindRef("refs", &new_sound_ref) == B_OK)
		{
			BEntry new_sound_entry(&new_sound_ref);
			BPath new_sound_path;
			if(new_sound_entry.GetPath(&new_sound_path) == B_NO_ERROR)
			{
				const char* new_sound = new_sound_path.Path();
				if (acquire_sem(g_sound_sem) == B_NO_ERROR)
				{
					if(g_sound_file)
						delete[] g_sound_file;
					g_sound_file = new char[strlen(new_sound)+1];
					strcpy(g_sound_file,new_sound);
					release_sem(g_sound_sem);
				}
				
			}
		}
	}
	else
		BView::MessageReceived(a_message);
}


MessageSoundEditTextView::MessageSoundEditTextView(BRect a_frame, MessageSoundDataHandler *a_attach_to)
: WrappingTextView(a_frame,NULL,B_FOLLOW_ALL_SIDES,B_WILL_DRAW)
{
	m_attached_to = a_attach_to;
	SetText(m_attached_to->GetMessage());
}


MessageSoundEditTextView::~MessageSoundEditTextView()
{ }


void MessageSoundEditTextView::StoreChange()
{
	m_attached_to->SetMessage(Text());
}
